<?php
/*
Plugin Name: Serverstatus
Description: Individueller Serverstatus mit Countdown, Statistik und Push-Benachrichtigungen
Version: 1.5
Author: TBG-Media
*/

// -------------------- Admin-Seite --------------------
function serverstatus_admin_page() {
    if (isset($_POST['pw']) && $_POST['pw'] === 'TBG') {
        $new_status = sanitize_text_field($_POST['status']);
        $note       = sanitize_text_field($_POST['note']);

        update_option('server_status', $new_status);
        update_option('server_note', $note);

        // Zeitstempel als GMT-Timestamp speichern für korrekte MEZ/MESZ
        update_option('server_last_update', current_time('timestamp', true));

        // Wartungs-Endzeit NUR speichern, wenn Status "wartung" ist und Feld gesetzt wurde
        if ($new_status === 'wartung' && !empty($_POST['maintenance_end'])) {
            $end_ts = strtotime(sanitize_text_field($_POST['maintenance_end']));
            if ($end_ts && $end_ts > time()) {
                update_option('server_maintenance_end', $end_ts);
            }
        } else {
            // Bei anderen Stati Wartungs-Endzeit entfernen
            delete_option('server_maintenance_end');
        }

        // Statistik aktualisieren (einfach gehalten)
        $stats = get_option('server_stats', ['offline_count'=>0,'total_changes'=>0]);
        $stats['total_changes']++;
        if ($new_status === 'offline') {
            $stats['offline_count']++;
        }
        update_option('server_stats', $stats);

        echo "<div class='notice notice-success is-dismissible'><p>Status gespeichert!</p></div>";
    } elseif (!empty($_POST)) {
        echo "<div class='notice notice-error is-dismissible'><p>Falsches Passwort!</p></div>";
    }

    $status = get_option('server_status', 'offline');
    $note   = get_option('server_note', '');
    $end    = get_option('server_maintenance_end', '');
    ?>
    <div class="wrap">
        <h1><span class="dashicons dashicons-admin-site"></span> Serverstatus ändern</h1>
        <div class="card">
            <form method="post">
                <p>
                    <label for="pw"><strong>Passwort:</strong></label><br>
                    <input type="password" id="pw" name="pw" class="regular-text">
                </p>

                <p>
                    <label for="status"><strong>Status:</strong></label><br>
                    <select id="status" name="status">
                        <option value="online" <?php selected($status,'online'); ?>>✅ Online</option>
                        <option value="wartung" <?php selected($status,'wartung'); ?>>🛠 Wartung</option>
                        <option value="teilweise" <?php selected($status,'teilweise'); ?>>⚠ Teilweise verfügbar</option>
                        <option value="offline" <?php selected($status,'offline'); ?>>❌ Offline</option>
                    </select>
                </p>

                <p>
                    <label for="note"><strong>Hinweis:</strong></label><br>
                    <input type="text" id="note" name="note" value="<?php echo esc_attr($note); ?>" class="regular-text">
                </p>

                <!-- Wartungs-Endzeit: nur sichtbar bei Status "Wartung" -->
                <p id="maintenance_wrap" style="display: none;">
                    <label for="maintenance_end"><strong>Wartung endet am (YYYY-MM-DD HH:MM):</strong></label><br>
                    <input type="text" id="maintenance_end" name="maintenance_end" value="<?php echo $end ? date('Y-m-d H:i', $end) : ''; ?>" class="regular-text" placeholder="2025-11-20 22:00">
                    <small>Nur relevant, wenn Status „Wartung“ aktiv ist.</small>
                </p>

                <p>
                    <input type="submit" value="Speichern" class="button button-primary button-large">
                </p>
            </form>
        </div>
    </div>

    <script>
    (function(){
        const select = document.getElementById('status');
        const wrap   = document.getElementById('maintenance_wrap');
        function toggle(){
            wrap.style.display = (select.value === 'wartung') ? 'block' : 'none';
        }
        toggle();
        select.addEventListener('change', toggle);
    })();
    </script>
    <?php
}

// Menüpunkt im Dashboard
function serverstatus_menu() {
    add_menu_page('Serverstatus', 'Serverstatus', 'manage_options', 'serverstatus', 'serverstatus_admin_page', 'dashicons-cloud', 80);
}
add_action('admin_menu', 'serverstatus_menu');

// -------------------- Font Awesome --------------------
function serverstatus_enqueue_icons() {
    wp_enqueue_style('font-awesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css');
}
add_action('wp_enqueue_scripts', 'serverstatus_enqueue_icons');

// -------------------- Shortcode --------------------
function serverstatus_shortcode() {
    $status      = get_option('server_status', 'offline');
    $note        = get_option('server_note', '');
    $last_update = get_option('server_last_update', ''); // GMT timestamp
    $end         = get_option('server_maintenance_end', ''); // UNIX timestamp

    $map = [
        'online'    => '<span style="background:#28a745;color:#fff;padding:6px 12px;border-radius:20px;font-weight:bold;"><i class="fas fa-check-circle" style="margin-right:6px;"></i>Online</span>',
        'wartung'   => '<span style="background:#ffc107;color:#000;padding:6px 12px;border-radius:20px;font-weight:bold;"><i class="fas fa-tools" style="margin-right:6px;"></i>Wartung</span>',
        'teilweise' => '<span style="background:#17a2b8;color:#fff;padding:6px 12px;border-radius:20px;font-weight:bold;"><i class="fas fa-exclamation-triangle" style="margin-right:6px;"></i>Teilweise verfügbar</span>',
        'offline'   => '<span style="background:#dc3545;color:#fff;padding:6px 12px;border-radius:20px;font-weight:bold;"><i class="fas fa-times-circle" style="margin-right:6px;"></i>Offline</span>',
    ];

    $html = "<div style='margin:15px 0;padding:15px;border:1px solid #444;border-radius:8px;background:#000;color:#fff;font-family:sans-serif;'>";
    $html .= "<p><strong>Status:</strong> ".$map[$status]."</p>";
    if ($note) $html .= "<p><strong>Hinweis:</strong> ".esc_html($note)."</p>";

    // Korrekte MEZ/MESZ Anzeige
    if ($last_update) {
        $tz = wp_timezone();
        $dt = new DateTime('@'.$last_update); // '@' interpretiert als UTC
        $dt->setTimezone($tz);
        $formatted = $dt->format('d.m.Y H:i');
        $suffix    = ($dt->format('I') === '1') ? 'MESZ' : 'MEZ'; // I = Daylight Saving Time (1/0)
        $html .= "<p style='font-size:0.9em;color:#ccc;'><em>Zuletzt aktualisiert am: ".$formatted." (".$suffix.")</em></p>";
    }

    // Countdown + Fortschrittsbalken nur bei aktiver Wartung und gesetztem Endzeitpunkt
    if ($status === 'wartung' && !empty($end) && $end > time()) {
        $html .= "<p><strong>Countdown:</strong> <span id='server-countdown'></span></p>";
        $html .= "<div style='width:100%;background:#333;border-radius:6px;overflow:hidden;height:20px;margin-top:5px;position:relative;'>
                    <div id='server-progress' style='height:100%;width:0;background:linear-gradient(90deg,#28a745,#ffc107,#dc3545);transition:width 0.5s ease;'></div>
                    <span id='server-percent' style='position:absolute;top:0;left:50%;transform:translateX(-50%);color:#fff;font-size:12px;font-weight:bold;'>0%</span>
                  </div>
                  <script>
                  (function(){
                      var end = new Date(<?php echo (int)$end * 1000; ?>).getTime();
                      var start = new Date().getTime();
                      var total = end - start;
                      var countdownEl = document.getElementById('server-countdown');
                      var progressEl = document.getElementById('server-progress');
                      var percentEl = document.getElementById('server-percent');
                      function pad(n){return n<10 ? '0'+n : n;}
                      function updateCountdown(){
                          var now = new Date().getTime();
                          var distance = end - now;
                          if(distance > 0){
                              var hours = Math.floor(distance / (1000*60*60));
                              var minutes = Math.floor((distance % (1000*60*60)) / (1000*60));
                              var seconds = Math.floor((distance % (1000*60)) / 1000);
                              countdownEl.textContent = hours + 'h ' + pad(minutes) + 'min ' + pad(seconds) + 's';
                              var percent = Math.max(0, Math.min(100, ((total - distance) / total) * 100));
                              progressEl.style.width = percent + '%';
                              percentEl.textContent = Math.floor(percent) + '%';
                          } else {
                              countdownEl.textContent = 'Wartung beendet – bitte Seite neu laden.';
                              progressEl.style.width = '100%';
                              progressEl.style.background = '#28a745';
                              percentEl.textContent = '100%';
                          }
                      }
                      updateCountdown();
                      setInterval(updateCountdown, 1000);
                  })();
                  </script>";
    }

    $html .= "</div>";
    return $html;
}
add_shortcode('serverstatus', 'serverstatus_shortcode');

// -------------------- Statistik-Dashboard --------------------
function serverstatus_stats_page() {
    $stats = get_option('server_stats', ['offline_count'=>0,'total_changes'=>0]);
    $offline_count = (int)$stats['offline_count'];
    $total_changes = (int)$stats['total_changes'];
    $uptime = $total_changes > 0 ? round((($total_changes - $offline_count)/$total_changes)*100,2) : 100;

    echo "<div class='wrap'><h1>📊 Serverstatistik</h1>";
    echo "<p><strong>Offline gezählt:</strong> {$offline_count} mal</p>";
    echo "<p><strong>Verfügbarkeit (Uptime):</strong> {$uptime}%</p>";
    echo "<p><em>Hinweis:</em> Die Uptime basiert auf einfachen Statusänderungen. Für exakte Werte könnte man Zeiträume protokollieren.</p>";
    echo "</div>";
}
function serverstatus_stats_menu() {
    add_menu_page('Serverstatistik','Serverstatistik','manage_options','serverstatus_stats','serverstatus_stats_page','dashicons-chart-bar',81);
}
add_action('admin_menu','serverstatus_stats_menu');

// -------------------- Push-Benachrichtigungen (Browser) --------------------
function serverstatus_push_notifications() {
    ?>
    <script>
    document.addEventListener("DOMContentLoaded", function(){
        if ("Notification" in window) {
            if (Notification.permission !== "granted") {
                Notification.requestPermission();
            }
            // Info-Notification beim Seitenbesuch mit aktuellem Status
            var status = "<?php echo esc_js(get_option('server_status','offline')); ?>";
            var note   = "<?php echo esc_js(get_option('server_note','')); ?>";
            var msg = "Serverstatus: " + status + (note ? " ("+note+")" : "");
            try {
                new Notification("🔔 Serverstatus", { body: msg });
            } catch(e) { /* still */ }
        }
    });
    </script>
    <?php
}
add_action('wp_footer','serverstatus_push_notifications');
